import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

/** Frame for the Board game. The frame contains the game
    board and necessary buttons and information fields. */
public class BoardGame extends JFrame {

   // Components that go in the frame
      Square[][] board; // Will contain the 8 x 8 game board
      Label whitesLabel= new Label("white pieces ",Label.CENTER);
      Label blacksLabel= new Label("black pieces ",Label.CENTER);
      Label whites= new Label("0",Label.CENTER); // Number of white pieces on board
      Label blacks= new Label("0",Label.CENTER); // Number of black pieces on board
      Label whoPlays= new Label("White to play ",Label.CENTER);
      TextArea helpText= null; // To contain an explanation, when requested
  
   // Titles for buttons
   private final String[] buttons= {"New game", "Quit", "Help"};
   
   // The layout and constraint variables for this frame
      GridBagLayout gb;
      GridBagConstraints gbc;

   int whiteCount= 0; 	// Number of white pieces on the board
   int blackCount= 0; 	// Number of black pieces on the board

   // These variables have values Square.WHITE or Square.BLACK
   int pieceToPlay; 	// piece to place next
   int pieceToTurn;	// piece to turn over if pieceToPlay is placed

   // Each of the eight pairs (horiz[d], vertic[d]) gives a direction
   // on the board.  E.g. For a square board[col][row],
   // board[col+horiz[d]][row+vertic[d]] is its neighbor in direction d.
   private final int[] horiz= {-1, 0, 1, 1, 1, 0,-1,-1}; 
   private final int[] vertic={-1,-1,-1, 0, 1, 1, 1, 0};

   
   /** Constructor: An 8 x 8 board, with three buttons and some
       pieces of information */
   public BoardGame() {
      super("Board Game");

      gb= new GridBagLayout();
      gbc= new GridBagConstraints();
      setFont(new Font("Dialog", Font.PLAIN, 10));
      getContentPane().setLayout(gb);
      gbc.fill  =  GridBagConstraints.NONE;
      int i;
      int j;
      
      // Create and place the 8*8 squares in the frame
         //System.out.println("Create the board");
         board= new Square[8][8];
         for (i= 0; i != 8; i= i+1)
            for (j= 0; j != 8; j= j+1) {
               board[i][j]= new Square(i,j);
               add(board[i][j], gb, gbc, 2*i,2*j, 2,2, 0,0);
               board[i][j].addMouseListener(new SquareClick());
            }

      // Create the buttons given by String buttons. Setting gbc.fill to BOTH
      // ensures that the buttons fill their entire space; without this
      // assignment, buttons would take the minimum space possible. Comment out
      // the first statement and execute to see this.
         gbc.fill=  GridBagConstraints.BOTH;
         Button newGameButton= new Button("New game");
         add(newGameButton, gb, gbc, 2*8,2*0, 2,2, 0,0);
         newGameButton.addActionListener(new NewGameButton());
         
         Button quitButton= new Button("Quit");
         add(quitButton, gb, gbc, 2*8,2*1, 2,2, 0,0);
         quitButton.addActionListener(new QuitButton());
      
         Button helpButton= new Button("Help");
         add(helpButton, gb, gbc, 2*8,2*2, 2,2, 0,0);
         helpButton.addActionListener(new HelpButton());

      // Place labels for the numbers of white and black pieces
         gbc.fill=  GridBagConstraints.HORIZONTAL;
         gbc.anchor= GridBagConstraints.SOUTHWEST;
         add(whitesLabel, gb, gbc, 2*8,2*4, 2,2, 0,0);
         add(blacksLabel, gb, gbc, 2*8,2*6, 2,2, 0,0);
         //gbc.anchor= GridBagConstraints.NORTHWEST;
         add(whites,      gb, gbc, 2*8,2*5, 2,2, 0,0);
         add(blacks,      gb, gbc, 2*8,2*7, 2,2, 0,0);

      // Place the "whose play is it" and cursor label
         add(whoPlays,    gb, gbc, 2*8,2*3, 2,2, 0,0);
         
     // add window closing event handler
         addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {
			   System.exit(0);}});
   
      pack();
      setLocation(150,50);
      newGame();
      show();
      setResizable(false);
   }

   // Add component c to gbl (with constraints gbc) at position (x,y).
   // Component c takes w columns and r rows, and is weighted (wx, wy).
   private void add(Component c, GridBagLayout gbl,
				    GridBagConstraints gbc,
                    int x, int y, int w, int h, int wx, int wy){
      gbc.gridx= x;
      gbc.gridy= y;
      gbc.gridwidth= w;
      gbc.gridheight= h;
      gbc.weightx= wx;
      gbc.weighty= wy;
      gbl.setConstraints(c, gbc);
      getContentPane().add(c);
   }

   /** Set the board up for a new game */
   public void newGame() {
      // Pick up the pieces from all the squares
         // PUT YOUR CODE HERE

      // Place the first four pieces
         // PUT YOUR CODE HERE

      whiteCount= 2;
      blackCount= 2;
      setCounts();
      pieceToPlay= Square.BLACK;
      pieceToTurn= Square.WHITE;
      changeTurn();
   }

   // Store whiteCount and blackCount in their respective textFields
   private void setCounts() {
      whites.setText(String.valueOf(whiteCount) + "  ");   
      blacks.setText(String.valueOf(blackCount) + "  ");
   }

   /** Process a click in square sq */
   public void processSquareClick(Square sq) {
      // Disable all the components of the board, so that
      // the players can't play until processing is over
         enableBoard(false);
   
      if (sq.contents() != Square.EMPTY) {
         // Square already occupied; clean up and return
            enableBoard(true);
            return;
      }

	  // Set numToTurn to the number of pieces that can be turned
	  // over if a piece x is placed in square sq
	     int numToTurn= 0;
	     for (int d= 0; d != 8; d= d+1) {
	        numToTurn= numToTurn + numberToTurn(d, sq.col, sq.row);
	     }

      if (numToTurn == 0) {
         // Can't place piece here; clean up and return
            enableBoard(true);
            return;
      }
      
      // A piece can be turned; turn pieces over in each of the 8 directions
         for (int d= 0; d != 8; d= d+1) {
            if (numberToTurn(d, sq.col, sq.row) > 0)
               turnOver(d, sq.col, sq.row);
         }

      putDown(pieceToPlay, sq.col, sq.row);
      changeTurn();
      if (cannotPlay())
         changeTurn();
      if (cannotPlay())
         whoPlays.setText("Game over");
      enableBoard(true);
   }

   /** If en is true, enable the board squares; otherwise disable them */
   public void enableBoard(boolean en) {
      for (int i= 0; i != 8; i= i+1)
         for (int j= 0; j != 8; j= j+1)
            board[i][j].setEnabled(en);
   }

   /** Yield "player pieceToPlay cannot make a play
       (because it cannot turn over a piece) */
   public boolean cannotPlay() {
      // For each EMPTY board position, return false if
      // there is a direction for which numberToTurn > 0
         for (int col= 0; col != 8; col= col+1)
            for (int row= 0; row != 8; row= row+1)
               // PUT YOUR CODE HERE. NOTE THAT ONLY EMPTY
               // SQUARES SHOULD BE PROCESSED
      return true;
   }

   /** Switch pieceToPlay and pieceToTurn frame */
   public void changeTurn() {
      int temp= pieceToPlay;
      pieceToPlay= pieceToTurn;
      pieceToTurn= temp;
      if (pieceToPlay == Square.WHITE) {
         whoPlays.setText("White to play");
         setCursor(new Cursor(Cursor.TEXT_CURSOR));
      }
      else {
         whoPlays.setText("Black to play");
         setCursor(new Cursor(Cursor.HAND_CURSOR));
      }
   }

   /** Yield the number of pieces that could be turned over in
       direction d if a piece pieceToPlay were placed on board[col][row]
       Thus, board[col][row] should be EMPTY when this method is called */
   public int numberToTurn(int d, int col, int row) {
      int adjcol= col + horiz[d];
      int adjrow= row + vertic[d];
      int numy= 0;
   
      // invariant: square board[adjcol][adjrow] is in direction d
      //    from board[col][row] (but it may be off the board). All the pieces
      //	   strictly between the two board positions board[col][row] and
      //    board[adjcol][adjrow] are pieceToTurn pieces, and there are numy of them
      while (onBoard(adjcol,adjrow) &&
            board[adjcol][adjrow].contents() == pieceToTurn) {
         // PUT YOUR CODE HERE
      }
      
      if ((!onBoard(adjcol,adjrow)) ||
            board[adjcol][adjrow].contents() != pieceToPlay)
         numy = 0;
      return numy;
   }

   /** At least one piece can be turned over in direction d, beginning at the
       position adjacent to board[col][row]. Turn them all over (in that
       direction). */
   public void turnOver(int d, int col, int row) {
      int adjcol= col + horiz[d];
      int adjrow= row + vertic[d];
      
      // invariant: square board[adjcol][adjrow] is in direction d
      //    from board[col][row], and all pieces strictly between the two
      //    board positions board[col][row] and board[adjcol][adjrow] were
      //    pieceToTurn and have been turned over
      while (board[adjcol][adjrow].contents() == pieceToTurn) {
         pickUp(adjcol,adjrow);
         putDown(pieceToPlay,adjcol, adjrow);
         adjcol= adjcol + horiz[d];
         adjrow= adjrow + vertic[d];
      }
   }

   /** Pick up the piece on board[col][row] */
   public void pickUp(int col, int row) {
      board[col][row].pickUpPiece();
      if (pieceToTurn == Square.WHITE)
         whiteCount= whiteCount-1;
      else
         blackCount= blackCount-1;

      setCounts();
   }

   /** board[col, row] is empty. Place piece p there
       (p is either Square.WHITE or Square.BLACK) */
   public void putDown(int p, int col, int row) {
      // Fill in this body
   }

   /** Yield "board[col][row] is a valid square of the board" */
   public boolean onBoard(int col, int row) {
      return  0 <= col && col < 8 && 0 <= row && row < 8;
   }
   
   private class SquareClick implements MouseListener {
		// Handle click of mouse
      public void mouseClicked(MouseEvent e) {
           processSquareClick((Square)(e.getComponent()));
           e.consume();
		}
		public void mouseEntered(MouseEvent e) {}
		public void mouseExited(MouseEvent e) {}
		public void mousePressed(MouseEvent e) {}
		public void mouseReleased(MouseEvent e) {}
	}
   
   private class NewGameButton implements ActionListener {
		// Handle press of "New game"
      public void actionPerformed(ActionEvent e) {
		   newGame();
		}
	}
	
	private class QuitButton implements ActionListener {
	   // Handle press of "Quit"
		public void actionPerformed(ActionEvent e) {
		   System.exit(0);
		}
	}
	
	private class HelpButton implements ActionListener {
	   // Handle press of "Help"
		public void actionPerformed(ActionEvent e) {
		   if (helpText != null) {
              // Help field already exists, so remove it and return
                 remove(helpText);
                 helpText= null;
                 pack();
                 setVisible(false);
                 setVisible(true);
                 return;
           }

           helpText= new TextArea("The object of the game is to win!" +
            "The game starts as given when you press \"New game\". There are two" +
            "\nplayers: white and black. White goes first, by placing a piece in some" +
            "\nempty square (do this by clicking the mouse in the square in which you" +
            "\n want to place a piece). However, you can't place a white piece in just" +
            "\nany square, but only in one that satisfies the following condition:" +
            "\n\n     In at least one direction (vertical, horizontal, or diagonal) emanating" +
            "\n     from the square, there is a sequence of one or more black pieces" +
            "\n     followed by a white piece." +
            "\n\nWhen a white piece is placed in such a square, all the black pieces" +
            "\nmentioned in this condition become white --in all directions. Try it!" +
            "\n\nAfter a white piece is placed, the other player places a black piece in the" +
            "\nsame manner (with the roles of white and black interchanged). Then it's" +
            "\nwhite's turn again, an so it goes." +
            "\n\nA player who cannot place a piece  loses their turn. If no player can play," +
            "\nthe game is over, and the player with the most pieces wins.");

           helpText.setEditable(false);
           gbc.fill= GridBagConstraints.BOTH;
           add(helpText, gb, gbc, 0,17, 18,1, 0,0);
           setSize(400,400);
           pack();
		}
	}

}
 
